<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Board;

use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Board\Events\BoardMovedToStage;
use Modules\Board\Models\Board;
use Modules\Board\Models\Stage;
use Modules\Users\Models\User;

class BoardUpdater
{
    /**
     * Caches the workable board
     *
     * @see board method
     */
    protected ?Collection $board = null;

    /**
     * Stages cache
     */
    protected Collection $stages;

    /**
     * The payload data
     */
    protected Collection $data;

    /**
     * Initialize new BoardUpdater instance.
     */
    public function __construct(array $data, protected User $user)
    {
        $this->data = collect($data);
        $this->stages = Stage::get();
    }

    /**
     * Performs the update
     */
    public function perform(): void
    {
        $data = $this->filterAuthorizedBoard();
        $order = $data->mapWithKeys(fn (array $data) => [$data['id'] => $data['board_order']]);

        // We will map the appropriate data for the Batch
        // so we can perform the update without any injected fields
        $attributes = $data->map(function (array $attrs) {
            $model = $this->board($attrs['id']);
            $stageId = (int) $attrs['stage_id'];

            $updatedAt = $model->stage_id === $stageId ? $model->updated_at : now();

            return [
                'id' => (int) $attrs['id'],
                'stage_id' => $stageId,
                'swatch_color' => $attrs['swatch_color'],
                'board_order' => $attrs['board_order'],
                'updated_at' => $updatedAt->format($model->getDateFormat()),
            ];
        })->all();

        $this->triggerMovedToStageEventIfNeeded($attributes);

        $this->update($attributes);

        $this->ensureNotVisibleBoardAreSortedAsLast($attributes[0]['stage_id'] ?? null, $order);
    }

    protected function ensureNotVisibleBoardAreSortedAsLast($stageId, $data): void
    {
        if (! $stageId) {
            return;
        }

        $max = $data->flatten()->max();
        $allIds = $data->keys()->all();

        Board::whereNotIn('id', $allIds)
            ->where('stage_id', $stageId)
            ->update(['board_order' => DB::raw("$max+board_order")]);
    }

    /**
     * Update the board from the payload
     *
     * If we change this method to not perform the update via batch,
     * check the BoardObserver because in the updated event the the LogBoardMovedToStageActivity
     * listener is triggered too
     */
    protected function update(array $data): void
    {
        $this->fireModelsEvent('updating', $data);
        batch()->update(new Board, $data);
        $this->fireModelsEvent('updated', $data);
    }

    /**
     * Get the board based on the id's provided in the payload|data
     */
    protected function board($id = null): Board|Collection
    {
        if (! $this->board) {
            $this->board = Board::with(['pipeline', 'user'])->findMany(
                $this->data->pluck('id')->all()
            );
        }

        if ($id) {
            return $this->board->find($id);
        }

        return $this->board;
    }

    /**
     * Trigger the board moved to stage event if needed
     */
    protected function triggerMovedToStageEventIfNeeded(array $board): void
    {
        foreach ($board as $data) {
            $board = $this->board($data['id']);

            if ($board->stage_id !== $data['stage_id']) {
                $oldStage = $this->stages->find($board->stage_id);

                // Update with the new stage data
                $board->setRelation('stage', $this->stages->find($data['stage_id']));
                $board->stage_id = $data['stage_id'];

                event(new BoardMovedToStage($board, $oldStage));
            }
        }
    }

    /**
     * Fire model events
     */
    protected function fireModelsEvent(string $event, array $data): void
    {
        foreach ($data as $attributes) {
            $board = $this->board($attributes['id']);

            $board->boardFiresEvents = true;

            if ($event === 'updating') {
                $board->forceFill($attributes);
            }

            $board->getEventDispatcher()->dispatch("eloquent.{$event}: ".$board::class, $board);

            $board->boardFiresEvents = false;
        }
    }

    /**
     * Remove any board which the user is not authorized to update
     */
    protected function filterAuthorizedBoard(): Collection
    {
        return $this->data->reject(
            fn ($data) => $this->user->cant('update', $this->board($data['id']))
        );
    }
}
