<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Http\Controllers\Api;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Modules\Core\Criteria\RequestCriteria;
use Modules\Core\Facades\Innoclapps;
use Modules\Core\Http\Controllers\ApiController;
use Modules\Board\Board\Board;
use Modules\Board\Criteria\ViewAuthorizedBoardCriteria;
use Modules\Board\Http\Resources\BoardBoardResource;
use Modules\Board\Models\Board;
use Modules\Board\Models\Pipeline;

class BoardBoardController extends ApiController
{
    /**
     * Get the board initial board data.
     */
    public function board(Pipeline $pipeline, Request $request): JsonResponse
    {
        $this->authorize('view', $pipeline);

        $pages = $request->input('pages', []);

        $stages = (new Board($request))->data($this->initialQuery(), (int) $pipeline->id, $pages);

        return $this->response(BoardBoardResource::collection($stages));
    }

    /**
     * Load more board for the given stage.
     */
    public function load(Pipeline $pipeline, string $stageId, Request $request): JsonResponse
    {
        $this->authorize('view', $pipeline);

        $stage = (new Board($request))->load($this->initialQuery(), (int) $stageId);

        return $this->response(new BoardBoardResource($stage));
    }

    /**
     * Update board card order and stage.
     */
    public function update(Pipeline $pipeline, Request $request): void
    {
        $this->authorize('view', $pipeline);

        $request->validate([
            // Must be present for adding/removing the color
            '*.swatch_color' => 'present|max:7',
            '*.id' => 'required',
            '*.stage_id' => 'required',
            '*.board_order' => 'required',
        ]);

        (new Board($request))->update($request->input());
    }

    /**
     * Get the board board summary for the given pipeline.
     */
    public function summary(Pipeline $pipeline, Request $request, ?string $stageId = null): JsonResponse
    {
        $this->authorize('view', $pipeline);

        return $this->response(
            (new Board($request))->summary($this->initialQuery(), (int) $pipeline->id, $stageId ?: null)
        );
    }

    protected function initialQuery()
    {
        return Board::criteria([
            ViewAuthorizedBoardCriteria::class,
            (new RequestCriteria)->setSearchFields(
                Innoclapps::resourceByModel(Board::class)->searchableColumns()
            ),
        ]);
    }
}
