<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Models;

use Illuminate\Broadcasting\PrivateChannel;
use Illuminate\Database\Eloquent\Attributes\ObservedBy;
use Illuminate\Database\Eloquent\BroadcastableModelEventOccurred;
use Illuminate\Database\Eloquent\BroadcastsEvents;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphToMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Collection;
use Modules\Activities\Concerns\HasActivities;
use Modules\Billable\Concerns\HasProducts;
use Modules\Calls\Concerns\HasCalls;
use Modules\Core\Common\Media\HasMedia;
use Modules\Core\Common\Timeline\HasTimeline;
use Modules\Core\Concerns\HasCreator;
use Modules\Core\Concerns\HasTags;
use Modules\Core\Concerns\HasUuid;
use Modules\Core\Concerns\LazyTouchesViaPivot;
use Modules\Core\Concerns\Prunable;
use Modules\Core\Contracts\Resources\Resourceable as ResourceableContract;
use Modules\Core\Facades\ChangeLogger;
use Modules\Core\Models\Model;
use Modules\Core\Resource\Resourceable;
use Modules\Core\Workflow\HasWorkflowTriggers;
use Modules\Board\Database\Factories\BoardFactory;
use Modules\Board\Enums\BoardStatus;
use Modules\Board\Observers\BoardObserver;
use Modules\Documents\Concerns\HasDocuments;
use Modules\MailClient\Concerns\HasEmails;
use Modules\Notes\Models\Note;

#[ObservedBy(BoardObserver::class)]
class Deal extends Model implements ResourceableContract
{
    use BroadcastsEvents,
        HasActivities,
        HasCalls,
        HasCreator,
        HasDocuments,
        HasEmails,
        HasFactory,
        HasMedia,
        HasProducts,
        HasTags,
        HasTimeline,
        HasUuid,
        HasWorkflowTriggers,
        LazyTouchesViaPivot,
        Prunable,
        Resourceable,
        SoftDeletes;

    /**
     * The attributes that aren't mass assignable.
     *
     * @var array<string>|bool
     */
    protected $guarded = [
        'created_by',
        'created_at',
        'updated_at',
        'owner_assigned_date',
        'next_activity_id',
        'stage_changed_date',
        'uuid',
    ];

    const TAGS_TYPE = 'board';

    /**
     * Attributes and relations to log changelog for the model
     *
     * @var array
     */
    protected static $changelogAttributes = [
        '*',
        'user.name',
        'pipeline.name',
    ];

    /**
     * Exclude attributes from the changelog
     *
     * @var array
     */
    protected static $changelogAttributeToIgnore = [
        'updated_at',
        'created_at',
        'created_by',
        'owner_assigned_date',
        'stage_changed_date',
        'swatch_color',
        'board_order',
        'status',
        'won_date',
        'lost_date',
        'lost_reason',
        'next_activity_id',
        'deleted_at',
        // Stage change are handled via custom pivot log events
        'stage_id',
    ];

    /**
     * Provides the relationships for the pivot logger
     *
     * [ 'main' => 'reverse']
     */
    protected static array $logPivotEventsOn = [
        'companies' => 'board',
        'contacts' => 'board',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'expected_close_date' => 'date',
        'stage_changed_date' => 'datetime',
        'owner_assigned_date' => 'datetime',
        'won_date' => 'datetime',
        'lost_date' => 'datetime',
        'status' => BoardStatus::class,
        'amount' => 'decimal:3',
        'stage_id' => 'int',
        'pipeline_id' => 'int',
        'user_id' => 'int',
        'board_order' => 'int',
        'created_by' => 'int',
        'web_form_id' => 'int',
        'next_activity_id' => 'int',
        'next_activity_date' => 'datetime',
    ];

    /**
     * Indicates whether the board "updating" and "updated" events are triggered via the board
     */
    public bool $boardFiresEvents = false;

    /**
     * Indicates whether to broadcast to the current user when the model is updated
     */
    public bool $broadcastToCurrentUser = false;

    /**
     * Check whether the falls behind the expected close date
     */
    protected function fallsBehindExpectedCloseDate(): Attribute
    {
        return Attribute::get(function () {
            if (! $this->expected_close_date || ! $this->isOpen()) {
                return false;
            }

            return $this->expected_close_date->isPast();
        });
    }

    /**
     * Determine whether the board is lost.
     */
    public function isLost(): bool
    {
        return $this->status === BoardStatus::lost;
    }

    /**
     * Determine whether the board is won.
     */
    public function isWon()
    {
        return $this->status === BoardStatus::won;
    }

    /**
     * Determine whether the board is open.
     */
    public function isOpen(): bool
    {
        return $this->status === BoardStatus::open;
    }

    /**
     * Check whether the status for the board can be changed.
     */
    public function isStatusLocked(BoardStatus $status): bool
    {
        // If it's the same status, it's not locked.
        if ($this->status === $status) {
            return false;
        }

        if ($this->isLost() || $this->isWon()) {
            return $status !== BoardStatus::open;
        }

        return false;
    }

    /**
     * Fill the board status.
     */
    public function fillStatus(BoardStatus $status, ?string $lostReason = null): static
    {
        $this->status = $status;

        if ($status === BoardStatus::lost) {
            $this->lost_reason = $lostReason;
        }

        return $this;
    }

    /**
     * Mark the board as lost
     */
    public function markAsLost(?string $reason = null): static
    {
        $this->fillStatus(BoardStatus::lost, $reason)->save();

        return $this;
    }

    /**
     * Mark the board as wont
     */
    public function markAsWon(): static
    {
        $this->fillStatus(BoardStatus::won)->save();

        return $this;
    }

    /**
     * Mark the board as open
     */
    public function markAsOpen(): static
    {
        $this->fillStatus(BoardStatus::open)->save();

        return $this;
    }

    /**
     * Log status change activity for the board
     *
     * @return \Modules\Core\Models\Changelog
     */
    public function logStatusChangeActivity(string $type, array $attrs = [])
    {
        return ChangeLogger::onModel($this, [
            'lang' => [
                'key' => 'board::board.timeline.'.$type,
                'attrs' => $attrs,
            ],
        ])->log();
    }

    /**
     * Get the pipeline that belongs to the board.
     */
    public function pipeline(): BelongsTo
    {
        return $this->belongsTo(\Modules\Board\Models\Pipeline::class);
    }

    /**
     * Get the stage that belongs to the board.
     */
    public function stage(): BelongsTo
    {
        return $this->belongsTo(Stage::class);
    }

    /**
     * Get the stages history the board has been.
     */
    public function stagesHistory(): BelongsToMany
    {
        return $this->belongsToMany(Stage::class, 'board_stages_history')
            ->withPivot(['id', 'entered_at', 'left_at'])
            ->using(\Modules\Board\Models\StageHistory::class)
            ->orderBy('entered_at', 'desc')
            ->as('history');
    }

    /**
     * Get the stages history with time in
     */
    public function timeInStages(): Collection
    {
        return $this->stagesHistory->groupBy('id')->map(function ($stages) {
            return $stages->reduce(function ($carry, $stage) {
                // If left_at is null, is using the current time
                $carry += $stage->history->entered_at->diffInSeconds($stage->history->left_at);

                return $carry;
            }, 0);
        });
    }

    /**
     * Start start history from the board current stage
     */
    public function startStageHistory(): static
    {
        $this->recordStageHistory($this->stage_id);

        return $this;
    }

    /**
     * Get the board last stage history
     */
    public function lastStageHistory(): ?Stage
    {
        return $this->stagesHistory()->first();
    }

    /**
     * Stop the board last stage timing
     */
    public function stopLastStageTiming(): static
    {
        $latest = $this->lastStageHistory();

        if ($latest && is_null($latest['history']['left_at'])) {
            $this->stagesHistory()
                ->wherePivot('id', $latest->history->id)
                ->updateExistingPivot($latest->history->stage_id, ['left_at' => now()]);
        }

        return $this;
    }

    /**
     * Record stage history
     */
    public function recordStageHistory(int $stageId): static
    {
        $this->stopLastStageTiming();
        $this->stagesHistory()->attach($stageId, ['entered_at' => now()]);

        return $this;
    }

    /**
     * Get all of the contacts that are associated with the board
     */
    public function contacts(): MorphToMany
    {
        return $this->morphedByMany(\Modules\Contacts\Models\Contact::class, 'boardable')
            ->withTimestamps()
            ->orderBy('boardables.created_at');
    }

    /**
     * Get all of the companies that are associated with the board
     */
    public function companies(): MorphToMany
    {
        return $this->morphedByMany(\Modules\Contacts\Models\Company::class, 'boardable')
            ->withTimestamps()
            ->orderBy('boardables.created_at');
    }

    /**
     * Get all of the notes for the board
     */
    public function notes(): MorphToMany
    {
        return $this->morphToMany(\Modules\Notes\Models\Note::class, 'noteable');
    }

    /**
     * Get the board owner
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(\Modules\Users\Models\User::class);
    }

    /**
     * Provide the total column to be updated whenever the billable is updated
     */
    public function totalColumn(): string
    {
        return 'amount';
    }

    /**
     * Get the channels that model events should broadcast on.
     *
     * @param  string  $event
     * @return \Illuminate\Broadcasting\Channel|array
     */
    public function broadcastOn($event)
    {
        // We will broadcast only when stage is changed when the update
        // comes from the custom broadcast event in "BoardUpdater"
        if ($this->boardFiresEvents && ! $this->isDirty('stage_id')) {
            return [];
        }

        // Currently only the updated event is used
        return match ($event) {
            'updated' => [new PrivateChannel($this)],
            default => null,
        };
    }

    /**
     * Broadcast to current user when updated
     */
    public function broadcastToCurrentUser(): static
    {
        $this->broadcastToCurrentUser = true;

        return $this;
    }

    /**
     * Create a new broadcastable model event for the model.
     *
     * @return \Illuminate\Database\Eloquent\BroadcastableModelEventOccurred
     */
    protected function newBroadcastableEvent(string $event)
    {
        $instance = (new BroadcastableModelEventOccurred(
            $this,
            $event
        ));

        if ($this->broadcastToCurrentUser) {
            return $instance;
        }

        return $instance->dontBroadcastToCurrentUser();
    }

    /**
     * Get the data to broadcast for the model.
     *
     * @param  string  $event
     */
    public function broadcastWith($event): array
    {
        return [];
    }

    /**
     * Scope a query to only include open board.
     */
    public function scopeWon(Builder $query): void
    {
        $query->where('status', BoardStatus::won);
    }

    /**
     * Scope a query to only include open board.
     */
    public function scopeOpen(Builder $query): void
    {
        $query->where('status', BoardStatus::open);
    }

    /**
     * Scope a query to only include lost board.
     */
    public function scopeLost(Builder $query): void
    {
        $query->where('status', BoardStatus::lost);
    }

    /**
     * Scope a query to only include closed board.
     */
    public function scopeClosed(Builder $query): void
    {
        $query->where(function (Builder $query) {
            $query->where('status', BoardStatus::won)->orWhere('status', BoardStatus::lost);
        });
    }

    /**
     * Scope a query to only include only board of the given pipeline.
     */
    public function scopeOfPipeline(Builder $query, Pipeline|int $pipeline): void
    {
        $query->where('pipeline_id', is_int($pipeline) ? $pipeline : $pipeline->getKey());
    }

    /**
     * Eager load the relations that are required for the front end response.
     */
    public function scopeWithCommon(Builder $query): void
    {
        $query
            ->withCount([
                'products' => fn (Builder $query) => $query->withoutGlobalScope('displayOrder'),
            ])
            ->with(['stagesHistory']);
    }

    /**
     * Purge the model data.
     */
    public function purge(): void
    {
        foreach (['emails', 'contacts', 'companies', 'activities', 'documents'] as $relation) {
            $this->{$relation}()->withTrashedIfUsingSoftDeletes()->detach();
        }

        $this->loadMissing('billable');

        if ($this->billable) {
            $this->billable->delete();
        }

        $this->loadMissing('notes')->notes->each(function (Note $note) {
            $note->delete();
        });
    }

    /**
     * Provide the related pivot relationships to touch.
     */
    protected function relatedPivotRelationsToTouch(): array
    {
        return ['companies', 'board'];
    }

    /**
     * Create a new factory instance for the model.
     */
    protected static function newFactory(): BoardFactory
    {
        return BoardFactory::new();
    }
}
