<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Observers;

use Modules\Core\Models\PinnedTimelineSubject;
use Modules\Board\Enums\BoardStatus;
use Modules\Board\Models\Board;
use Modules\Board\Models\Stage;

class BoardObserver
{
    /**
     * Handle the Board "saving" event.
     */
    public function saving(Board $board): void
    {
        if ($board->isDirty('status')) {
            if ($board->isOpen()) {
                $board->fill(['won_date' => null, 'lost_date' => null, 'lost_reason' => null]);
            } elseif ($board->isLost()) {
                $board->fill(['lost_date' => now(), 'won_date' => null]);
            } elseif ($board->isWon()) {
                $board->fill(['won_date' => now(), 'lost_date' => null, 'lost_reason' => null]);
            }
        }

        if ($board->isDirty(['pipeline_id', 'stage_id'])) {
            $board->pipeline_id = Stage::findFromObjectCache($board->stage_id)->getAttribute('pipeline_id');
        }
    }

    /**
     * Handle the Board "creating" event.
     */
    public function creating(Board $board): void
    {
        if (! $board->status) {
            $board->status = BoardStatus::open;
        }
    }

    /**
     * Handle the Board "created" event.
     */
    public function created(Board $board): void
    {
        if ($board->isOpen()) {
            $board->startStageHistory();
        }

        // When new board is created, always increment the board order so the new
        // board is pushed on top and all board are ordered properly
        if ($board->board_order === 0) { // via tests?
            Board::withoutEvents(fn () => $board->newQuery()->increment('board_order'));
        }
    }

    /**
     * Handle the Board "updating" event.
     */
    public function updating(Board $board): void
    {
        if ($board->isDirty('stage_id')) {
            $board->stage_changed_date = now();
        }

        if (! $board->isDirty('status')) {
            // Guard these attributes when the status is not changed
            foreach (['won_date', 'lost_date'] as $guarded) {
                if ($board->isDirty($guarded)) {
                    $board->fill([$guarded => $board->getOriginal($guarded)]);
                }
            }

            // Allow updating the lost reason only when status is lost
            if (! $board->isLost() && $board->isDirty('lost_reason')) {
                $board->fill(['lost_reason' => $board->getOriginal('lost_reason')]);
            }
        }
    }

    /**
     * Handle the Board "updated" event.
     */
    public function updated(Board $board): void
    {
        if ($board->wasChanged('status')) {
            $changelog = $board->logStatusChangeActivity(
                'marked_as_'.$board->status->name,
                $board->isLost() ? [
                    'reason' => $board->lost_reason,
                ] : []
            );

            if ($changelog && $board->isLost()) {
                (new PinnedTimelineSubject)->pin(
                    $board->id, $board::class, $changelog->getKey(), $changelog::class
                );
            }

            if ($board->isOpen()) {
                $board->startStageHistory();
            } else {
                // Changed to won or lost
                $board->stopLastStageTiming();
            }
        }

        if ($board->wasChanged('stage_id') && $board->isOpen()) {
            $board->recordStageHistory($board->stage_id);
        }
    }

    /**
     * Handle the Board "deleting" event.
     */
    public function deleting(Board $board): void
    {
        if ($board->isForceDeleting()) {
            $board->purge();
        }
    }
}
