<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Policies;

use Illuminate\Auth\Access\HandlesAuthorization;
use Modules\Board\Models\Board;
use Modules\Users\Models\User;

class BoardPolicy
{
    use HandlesAuthorization;

    /**
     * Determine whether the user can view any board.
     */
    public function viewAny(User $user): bool
    {
        return true;
    }

    /**
     * Determine whether the user can view the board.
     */
    public function view(User $user, Board $board): bool
    {
        if ($user->can('view all board')) {
            return true;
        }

        if ((int) $board->user_id === (int) $user->id) {
            return true;
        }

        if ($board->user_id && $user->can('view team board')) {
            return $user->managesAnyTeamsOf($board->user_id);
        }

        return false;
    }

    /**
     * Determine if the given user can create board.
     */
    public function create(User $user): bool
    {
        return true;
    }

    /**
     * Determine whether the user can update the board.
     */
    public function update(User $user, Board $board): bool
    {
        if ($user->can('edit all board')) {
            return true;
        }

        if ($user->can('edit own board') && (int) $user->id === (int) $board->user_id) {
            return true;
        }

        if ($board->user_id && $user->can('edit team board') && $user->managesAnyTeamsOf($board->user_id)) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user can delete the board.
     */
    public function delete(User $user, Board $board): bool
    {
        if ($user->can('delete any board')) {
            return true;
        }

        if ($user->can('delete own board') && (int) $user->id === (int) $board->user_id) {
            return true;
        }

        if ($board->user_id && $user->can('delete team board') && $user->managesAnyTeamsOf($board->user_id)) {
            return true;
        }

        return false;
    }

    /**
     * Determine whether the user bulk delete board.
     */
    public function bulkDelete(User $user, ?Board $board = null)
    {
        if (! $board) {
            return $user->can('bulk delete board');
        }

        if ($board && $user->can('bulk delete board')) {
            return $this->delete($user, $board);
        }

        return false;
    }

    /**
     * Determine whether the user can export board.
     */
    public function export(User $user): bool
    {
        return $user->can('export board');
    }
}
