<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Providers;

use Illuminate\Support\Facades\Auth;
use Modules\Core\Database\State\DatabaseState;
use Modules\Core\Facades\Innoclapps;
use Modules\Core\Facades\Menu;
use Modules\Core\Settings\DefaultSettings;
use Modules\Core\Support\ModuleServiceProvider;
use Modules\Core\Workflow\Workflows;
use Modules\Board\Events\BoardMovedToStage;
use Modules\Board\Http\Resources\LostReasonResource;
use Modules\Board\Http\Resources\PipelineResource;
use Modules\Board\Listeners\LogBoardMovedToStageActivity;
use Modules\Board\Listeners\TransferBoardUserData;
use Modules\Board\Menu\OpenBoardMetric;
use Modules\Board\Models\Board;
use Modules\Board\Models\LostReason;
use Modules\Board\Models\Pipeline;
use Modules\Users\Events\TransferringUserData;

class BoardServiceProvider extends ModuleServiceProvider{
    protected array $resources = [
        \Modules\Board\Resources\Board::class,
        \Modules\Board\Resources\Pipeline::class,
        \Modules\Board\Resources\PipelineStage::class,
        \Modules\Board\Resources\LostReason::class,
    ];

    protected array $mailableTemplates = [
        \Modules\Board\Mail\UserAssignedToBoard::class,
    ];

    protected array $notifications = [
        \Modules\Board\Notifications\UserAssignedToBoard::class,
    ];

    /**
     * Bootstrap any module services.
     */
    public function boot(): void
    {
        $this->app['events']->listen(BoardMovedToStage::class, LogBoardMovedToStageActivity::class);
        $this->app['events']->listen(TransferringUserData::class, TransferBoardUserData::class);
    }

    /**
     * Register any module services.
     */
    public function register(): void
    {
        $this->app->register(RouteServiceProvider::class);
    }

    /**
     * Configure the module.
     */
    protected function setup(): void
    {
        $this->registerWorkflowTriggers();

        DatabaseState::register([
            \Modules\Board\Database\State\EnsureDefaultPipelineIsPresent::class,
        ]);

        DefaultSettings::add('allow_lost_reason_enter', true);
        DefaultSettings::add('lost_reason_is_required', true);

        Menu::metric(new OpenBoardMetric);

        Innoclapps::whenReadyForServing(function () {
            Innoclapps::booting($this->shareDataToScript(...));
        });
    }

    /**
     * Register the documents module available workflows.
     */
    protected function registerWorkflowTriggers(): void
    {
        Workflows::triggers([
            \Modules\Board\Workflow\Triggers\BoardCreated::class,
            \Modules\Board\Workflow\Triggers\BoardStageChanged::class,
            \Modules\Board\Workflow\Triggers\BoardStatusChanged::class,
        ]);
    }

    /**
     * Share module data to script.
     */
    protected function shareDataToScript(): void
    {
        Innoclapps::provideToScript(fn () => Auth::check() ? [
            'allow_lost_reason_enter' => settings('allow_lost_reason_enter'),
            'lost_reason_is_required' => settings('lost_reason_is_required'),
            'board' => [
                'tags_type' => Board::TAGS_TYPE,
                'pipelines' => PipelineResource::collection(
                    Pipeline::withCommon()
                        ->with('stages')
                        ->withVisibilityGroups()
                        ->visible()
                        ->orderByUserSpecified(Auth::user())
                        ->get()
                ),
                'lost_reasons' => LostReasonResource::collection(
                    LostReason::withCommon()->orderBy('name')->get()
                ),
            ],
        ] : []);
    }

    /**
     * Provide the module name.
     */
    protected function moduleName(): string
    {
        return 'Board';
    }

    /**
     * Provide the module name in lowercase.
     */
    protected function moduleNameLower(): string
    {
        return 'board';
    }
}
