<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Resources;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Response;
use Illuminate\Support\Carbon;
use Illuminate\Validation\Rule;
use Modules\Activities\Actions\CreateRelatedActivityAction;
use Modules\Activities\Fields\NextActivityDate;
use Modules\Activities\Filters\ResourceActivitiesFilter;
use Modules\Billable\Contracts\BillableResource;
use Modules\Billable\Fields\Amount;
use Modules\Billable\Filters\BillableProductsFilter;
use Modules\Billable\Services\BillableService;
use Modules\Comments\Contracts\PipesComments;
use Modules\Contacts\Fields\Companies;
use Modules\Contacts\Fields\Contacts;
use Modules\Core\Actions\Action;
use Modules\Core\Actions\DeleteAction;
use Modules\Core\Contracts\Resources\AcceptsCustomFields;
use Modules\Core\Contracts\Resources\Exportable;
use Modules\Core\Contracts\Resources\Importable;
use Modules\Core\Contracts\Resources\Mediable;
use Modules\Core\Contracts\Resources\Tableable;
use Modules\Core\Contracts\Resources\WithResourceRoutes;
use Modules\Core\Facades\Fields;
use Modules\Core\Facades\Innoclapps;
use Modules\Core\Fields\ColorSwatch;
use Modules\Core\Fields\CreatedAt;
use Modules\Core\Fields\Date;
use Modules\Core\Fields\DateTime;
use Modules\Core\Fields\ID;
use Modules\Core\Fields\RelationshipCount;
use Modules\Core\Fields\Tags;
use Modules\Core\Fields\Text;
use Modules\Core\Fields\UpdatedAt;
use Modules\Core\Fields\User;
use Modules\Core\Filters\CreatedAt as CreatedAtFilter;
use Modules\Core\Filters\Date as DateFilter;
use Modules\Core\Filters\DateTime as DateTimeFilter;
use Modules\Core\Filters\Filter;
use Modules\Core\Filters\FilterChildGroup;
use Modules\Core\Filters\FilterGroups;
use Modules\Core\Filters\HasFilter;
use Modules\Core\Filters\MultiSelect as MultiSelectFilter;
use Modules\Core\Filters\Numeric as NumericFilter;
use Modules\Core\Filters\Operand;
use Modules\Core\Filters\OperandFilter;
use Modules\Core\Filters\Select as SelectFilter;
use Modules\Core\Filters\Tags as TagsFilter;
use Modules\Core\Filters\Text as TextFilter;
use Modules\Core\Filters\UpdatedAt as UpdatedAtFilter;
use Modules\Core\Http\Requests\ActionRequest;
use Modules\Core\Http\Requests\ResourceRequest;
use Modules\Core\Menu\MenuItem;
use Modules\Core\Models\Model;
use Modules\Core\Pages\Panel;
use Modules\Core\Pages\Tab;
use Modules\Core\Resource\Import\Import;
use Modules\Core\Resource\Resource;
use Modules\Core\Rules\StringRule;
use Modules\Core\Settings\SettingsMenuItem;
use Modules\Core\Table\Column;
use Modules\Core\Table\Table;
use Modules\Board\Criteria\ViewAuthorizedBoardCriteria;
use Modules\Board\Enums\BoardStatus;
use Modules\Board\Events\BoardMovedToStage;
use Modules\Board\Fields\LostReasonField;
use Modules\Board\Filters\BoardStatusFilter;
use Modules\Board\Http\Resources\BoardResource;
use Modules\Board\Models\Board as BoardModel;
use Modules\Board\Models\Stage;
use Modules\Documents\Filters\ResourceDocumentsFilter;
use Modules\MailClient\Filters\ResourceEmailsFilter;
use Modules\Notes\Fields\ImportNote;
use Modules\Users\Filters\ResourceUserTeamFilter;
use Modules\Users\Filters\UserFilter;
use Modules\WebForms\Models\WebForm;

class Board extends Resource implements AcceptsCustomFields, BillableResource, Exportable, Importable, Mediable, PipesComments, Tableable, WithResourceRoutes
{
    /**
     * Indicates whether the resource has Zapier hooks
     */
    public static bool $hasZapierHooks = true;

    /**
     * The column the records should be default ordered by when retrieving
     */
    public static string $orderBy = 'name';

    /**
     * Indicates whether the resource has detail view.
     */
    public static bool $hasDetailView = true;

    /**
     * Indicates whether the resource is globally searchable
     */
    public static bool $globallySearchable = true;

    /**
     * Indicates the global search action. (view, float)
     */
    public static string $globalSearchAction = 'float';

    /**
     * The resource displayable icon.
     */
    public static ?string $icon = 'Chat';

    /**
     * Indicates whether the resource fields are customizeable
     */
    public static bool $fieldsCustomizable = true;

    public static string $model = 'Modules\Board\Models\Board';

    /**
     * The attribute to be used when the resource should be displayed.
     */
    public static string $title = 'name';

    /**
     * Get the menu items for the resource
     */
      public function menu(): array
    {
        return [
            MenuItem::make(static::label(), '/board', static::$icon)->position(20),
        ];
    }

    /**
     * Get the resource relationship name when it's associated
     */
    public function associateableName(): string
    {
        return 'board';
    }

    /**
     * Get the resource available cards
     */


    /**
     * Provide the resource table class instance.
     */
    public function table(Builder $query, ResourceRequest $request, string $identifier): Table
    {

        return BoardTable::make($query, $request, $identifier)
            ->withDefaultView(
                name: 'board::board.views.all',
                flag: 'all-board',
            )
            ->withDefaultView(
                name: 'board::board.views.my',
                flag: 'my-board',
                rules: new FilterGroups(new FilterChildGroup(rules: [
                    UserFilter::make()->setOperator('equal')->setValue('me'),
                ], quick: true))
            )
            ->withDefaultView(
                name: 'board::board.views.my_recently_assigned',
                flag: 'my-recently-assigned-board',
                rules: new FilterGroups([
                    new FilterChildGroup(rules: [
                        DateTimeFilter::make('owner_assigned_date')->setOperator('is')->setValue('this_month'),
                    ]),
                    new FilterChildGroup(rules: [
                        UserFilter::make()->setOperator('equal')->setValue('me'),
                    ], quick: true),
                ])
            )
            ->withDefaultView(
                name: 'board::board.views.created_this_month',
                flag: 'board-created-this-month',
                rules: new FilterGroups(new FilterChildGroup(rules: [
                    DateTimeFilter::make('created_at')->setOperator('is')->setValue('this_month'),
                ], quick: true))
            )
            ->withDefaultView(
                name: 'board::board.views.won',
                flag: 'won-board',
                rules: new FilterGroups(new FilterChildGroup(rules: [
                    BoardStatusFilter::make()->setOperator('equal')->setValue(BoardStatus::won->name),
                ], quick: true))
            )
            ->withDefaultView(
                name: 'board::board.views.lost',
                flag: 'lost-board',
                rules: new FilterGroups(new FilterChildGroup(rules: [
                    BoardStatusFilter::make()->setOperator('equal')->setValue(BoardStatus::lost->name),
                ], quick: true))
            )
            ->withDefaultView(
                name: 'board::board.views.open',
                flag: 'open-board',
                rules: new FilterGroups(new FilterChildGroup(rules: [
                    BoardStatusFilter::make()->setOperator('equal')->setValue(BoardStatus::open->name),
                ], quick: true))
            )
            ->orderBy('created_at', 'desc')
            ->rowBorderVariant(function (array $row) {
                return $row['falls_behind_expected_close_date'] ? 'warning' : null;
            });

    }

    public function jsonResource(): string
    {
        return BoardResource::class;
    }

    public function viewAuthorizedRecordsCriteria(): string
    {
        return ViewAuthorizedBoardCriteria::class;
    }

    public function fields(ResourceRequest $request): array
    {
        return [
            ID::make()->hidden(),

            Text::make('name', __('board::fields.board.name'))
                ->primary()
                ->tapIndexColumn(fn (Column $column) => $column
                    ->width('300px')->minWidth('200px')
                    ->primary()
                    ->route(! $column->isForTrashedTable() ? '/board/{id}' : '')
                )
                ->rules(StringRule::make())
                ->creationRules('required')
                ->updateRules('filled')
                ->importRules('required')
                ->hideFromDetail()
                ->excludeFromSettings(Fields::DETAIL_VIEW)
                ->required(true),


            Amount::make('amount', __('board::fields.board.amount'))
                ->readonly(fn () => $this->for?->hasProducts() ?? false)
                ->primary()
                ->currency()
                ->allowMinus(),

            Date::make('expected_close_date', __('board::fields.board.expected_close_date'))
                ->primary()
                ->clearable()
                ->withDefaultValue(Carbon::now()->endOfMonth()->format('Y-m-d')),

            Tags::make()
                ->forType(BoardModel::TAGS_TYPE)
                ->rules(['sometimes', 'nullable', 'array'])
                ->hideFromDetail()
                ->hideFromIndex()
                ->excludeFromSettings(Fields::DETAIL_VIEW),

            Text::make('status', __('board::board.status.status'))
                ->excludeFromSettings()
                ->hideFromDetail()
                ->hideWhenCreating()
                ->hideWhenUpdating()
                ->excludeFromImport()
                ->fillUsing(function (Model $model, string $attribute, ResourceRequest $request, mixed $value, string $requestAttribute) {
                    $status = BoardStatus::find($value);

                    abort_if(
                        $model->isStatusLocked($status),
                        Response::HTTP_CONFLICT,
                        'The board first must be marked as open in order to apply the "'.$status->name.'" status.'
                    );

                    $model->fillStatus($status, $request->lost_reason);
                })
                ->rules(['sometimes', 'nullable', 'string', Rule::in(BoardStatus::names())])
                ->showValueWhenUnauthorizedToView()
                ->resolveUsing(fn ($model) => $model->status->name)
                ->displayUsing(fn ($model, $value) => BoardStatus::find($value)->label()) // For mail placeholder
                ->tapIndexColumn(function (Column $column) {
                    $column->centered()
                        ->withMeta([
                            'statuses' => collect(BoardStatus::cases())->mapWithKeys(function ($status) {
                                return [$status->value => [
                                    'name' => $status->name,
                                    'badge' => $status->badgeVariant(),
                                ]];
                            }),
                        ])
                        ->orderByUsing(function (Builder $query, string $direction) {
                            return $query->orderByRaw('CASE
                                WHEN status ="'.BoardStatus::open->value.'" THEN 1
                                WHEN status ="'.BoardStatus::lost->value.'" THEN 2
                                WHEN status ="'.BoardStatus::won->value.'" THEN 3
                            END '.$direction);
                        });
                }),

            LostReasonField::make('lost_reason', __('board::board.lost_reasons.lost_reason'))
                ->hidden()
                ->excludeFromSettings()
                ->excludeFromImportSample()
                ->disableInlineEdit()
                ->rules(array_filter([
                    Rule::excludeIf(fn () => $request->resourceId()
                         && $request->record()->isLost() &&
                         $request->missing('lost_reason')
                    ),
                    (bool) settings('lost_reason_is_required') ? 'required_if:status,lost' : null,
                    'nullable',
                    StringRule::make(),
                ])),

            User::make(__('board::fields.board.user.name'))
                ->primary()
                ->acceptLabelAsValue(false)
                ->withMeta(['attributes' => ['placeholder' => __('core::app.no_owner')]])
                ->notification(\Modules\Board\Notifications\UserAssignedToBoard::class)
                ->trackChangeDate('owner_assigned_date')
                ->hideFromDetail()
                ->excludeFromSettings(Fields::DETAIL_VIEW)
                ->showValueWhenUnauthorizedToView(),

            Contacts::make()
                ->excludeFromSettings(Fields::DETAIL_VIEW)
                ->hideFromDetail()
                ->hideFromIndex()
                ->order(1001),

            Companies::make()
                ->excludeFromSettings(Fields::DETAIL_VIEW)
                ->hideFromDetail()
                ->hideFromIndex()
                ->order(1002),

            // API usage
            ColorSwatch::make('swatch_color', __('core::app.colors.color'))
                ->hidden()
                ->excludeFromSettings()
                ->excludeFromImportSample()
                ->excludeFromIndex(),

            DateTime::make('owner_assigned_date', __('board::fields.board.owner_assigned_date'))
                ->exceptOnForms()
                ->excludeFromSettings()
                ->hidden(),

            RelationshipCount::make('contacts', __('contacts::contact.total'))->hidden(),

            RelationshipCount::make('companies', __('contacts::company.total'))->hidden(),

            RelationshipCount::make('unreadEmailsForUser', __('mailclient::inbox.unread_count'))
                ->hidden()
                ->authRequired()
                ->excludeFromZapierResponse(),

            RelationshipCount::make('incompleteActivitiesForUser', __('activities::activity.incomplete_activities'))
                ->hidden()
                ->authRequired()
                ->excludeFromZapierResponse(),

            RelationshipCount::make('documents', __('documents::document.total_documents'))
                ->hidden()
                ->excludeFromZapierResponse(),

            RelationshipCount::make('draftDocuments', __('documents::document.total_draft_documents'))
                ->hidden()
                ->excludeFromZapierResponse(),

            RelationshipCount::make('calls', __('calls::call.total_calls'))->hidden(),

            NextActivityDate::make(),

            ImportNote::make(),

            CreatedAt::make()->hidden(),

            UpdatedAt::make()->hidden(),
        ];
    }

    public function importable(): Import
    {
        return new BoardImport($this);
    }

    public function filters(ResourceRequest $request): array
    {
        return [
            TextFilter::make('name', __('board::fields.board.name'))->withoutNullOperators(),
            NumericFilter::make('amount', __('board::fields.board.amount')),

            MultiSelectFilter::make('stage_id', __('board::fields.board.stage.name'))
                ->labelKey('name')
                ->valueKey('id')
                ->options(fn () => Stage::allStagesForOptions($request->user())),

            TagsFilter::make('tags', __('core::tags.tags'))->forType(BoardModel::TAGS_TYPE)->inQuickFilter(multiple: true),

            DateTimeFilter::make('won_date', __('board::board.won_date'))
                ->help(__('board::board.status_related_filter_notice', ['status' => BoardStatus::won->label()])),

            DateTimeFilter::make('lost_date', __('board::board.lost_date'))
                ->help(__('board::board.status_related_filter_notice', ['status' => BoardStatus::lost->label()])),

            BoardStatusFilter::make()->inQuickFilter(),

            DateFilter::make('expected_close_date', __('board::fields.board.expected_close_date')),

            TextFilter::make('lost_reason', __('board::board.lost_reasons.lost_reason')),

            UserFilter::make(__('board::fields.board.user.name'))->inQuickFilter(),
            ResourceUserTeamFilter::make(__('users::team.owner_team')),
            DateTimeFilter::make('owner_assigned_date', __('board::fields.board.owner_assigned_date')),
            DateTimeFilter::make('stage_changed_date', __('board::board.stage.changed_date')),
            ResourceDocumentsFilter::make(),
            BillableProductsFilter::make(),
            ResourceActivitiesFilter::make(),
            ResourceEmailsFilter::make(),

            SelectFilter::make('web_form_id', __('webforms::form.form'))
                ->labelKey('title')
                ->valueKey('id')
                ->options(function () {
                    return WebForm::get(['id', 'title'])->map(fn (WebForm $webForm) => [
                        'id' => $webForm->id,
                        'title' => $webForm->title,
                    ]);
                })
                ->canSeeWhen('is-super-admin'),

            DateTimeFilter::make('next_activity_date', __('activities::activity.next_activity_date')),
            UserFilter::make(__('core::app.created_by'), 'created_by')->withoutNullOperators()->canSeeWhen('view all board'),
            CreatedAtFilter::make()->inQuickFilter(),
            UpdatedAtFilter::make(),

            HasFilter::make('contacts', __('contacts::contact.contact'))->setOperands(
                fn () => Innoclapps::resourceByName('contacts')
                    ->resolveFilters($request)
                    ->reject(fn (Filter $filter) => $filter instanceof OperandFilter)
                    ->map(fn (Filter $filter) => Operand::from($filter))
                    ->values()
                    ->all()
            ),

            HasFilter::make('companies', __('contacts::company.company'))->setOperands(
                fn () => Innoclapps::resourceByName('companies')
                    ->resolveFilters($request)
                    ->reject(fn (Filter $filter) => $filter instanceof OperandFilter)
                    ->map(fn (Filter $filter) => Operand::from($filter))
                    ->values()
                    ->all()
            ),
        ];
    }

    /**
     * Provides the resource available actions
     */
    public function actions(ResourceRequest $request): array
    {
        return [
            (new \Modules\Board\Actions\MarkAsWon)->withoutConfirmation(),
            new \Modules\Board\Actions\MarkAsLost,
            (new \Modules\Board\Actions\MarkAsOpen)->withoutConfirmation(),

            new \Modules\Core\Actions\BulkEditAction($this),

            new \Modules\Board\Actions\ChangeBoardStage,

            CreateRelatedActivityAction::make()->onlyInline(),

            Action::make()->floatResourceInEditMode(),

            DeleteAction::make()->canRun(function (ActionRequest $request, Model $model, int $total) {
                return $request->user()->can($total > 1 ? 'bulkDelete' : 'delete', $model);
            })->showInline()->withSoftDeletes(),
        ];
    }

    /**
     * Prepare display query.
     */
    public function displayQuery(): Builder
    {
        return parent::displayQuery()->with([
            'pipeline.stages',
            'media',
            'contacts.phones', // phones are for calling
            'companies.phones', // phones are for calling
        ]);
    }

    /**
     * Prepare global search query.
     */
    public function globalSearchQuery(ResourceRequest $request): Builder
    {
        return parent::globalSearchQuery($request)->select(['id', 'name', 'created_at']);
    }

    /**
     * Get the displayable label of the resource
     */
    public static function label(): string
    {
        return 'Support Board';
    }

    /**
     * Get the displayable singular label of the resource
     */
    public static function singularLabel(): string
    {
        return 'Support Board';
    }

    /**
     * Register permissions for the resource
     */
    public function registerPermissions(): void
    {
        $this->registerCommonPermissions();

        Innoclapps::permissions(function ($manager) {
            $manager->group($this->name(), function ($manager) {
                $manager->view('export', [
                    'permissions' => [
                        'export board' => __('core::app.export.export'),
                    ],
                ]);
            });
        });
    }

    /**
     * Register the settings menu items for the resource
     */
    public function settingsMenu(): array
    {
        return [
            SettingsMenuItem::make(__('board::board.board'), '/settings/board', 'Folder')->order(22),
        ];
    }

    /**
     * Boot the resource.
     */
    protected function boot(): void
    {
        $this->getDetailPage()->tabs([
            Tab::make('timeline', 'timeline-tab')->panel('timeline-tab-panel')->order(10),
        ])->panels([
            Panel::make('board-detail-panel', 'resource-details-panel')
                ->heading(__('core::app.record_view.sections.details'))
                ->resizeable(),
            Panel::make('contacts', 'resource-contacts-panel')->heading(__('contacts::contact.contacts')),
            Panel::make('companies', 'resource-companies-panel')->heading(__('contacts::company.companies')),
            Panel::make('media', 'resource-media-panel')->heading(__('core::app.attachments')),
        ]);
    }

    /**
     * Handle the "afterCreate" resource record hook.
     */
    public function afterCreate(Model $model, ResourceRequest $request): void
    {
        if (count($request->billable['products'] ?? []) > 0) {
            (new BillableService)->save($request->billable, $model);
        }
    }

    /**
     * Handle the "beforeUpdate" resource record hook.
     */
    public function beforeUpdate(Model $model, ResourceRequest $request): void
    {
        if ($model->isDirty('stage_id')) {
            $request->merge(['_original_stage' => $model->getOriginal('stage_id')]);
        }
    }

    /**
     * Handle the "afterUpdate" resource record hook.
     */
    public function afterUpdate(Model $model, ResourceRequest $request): void
    {
        if ($request->billable) {
            (new BillableService)->save($request->billable, $model);
        }

        if ($model->wasChanged('stage_id')) {
            BoardMovedToStage::dispatch(
                $model,
                Stage::findFromObjectCache($request->input('_original_stage'))
            );
        }
    }
}
