<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Modules\Core\Tests\ResourceTestCase;
use Modules\Board\Models\Pipeline;
use Modules\Board\Models\Stage;
use Modules\Users\Models\User;

class ChangeBoardStageTest extends ResourceTestCase
{
    protected $action = 'change-board-stage';

    protected $resourceName = 'board';

    public function test_super_admin_user_can_run_change_board_stage_action(): void
    {
        $this->signIn();
        $pipeline = Pipeline::factory()->withStages()->create();
        $board = $this->factory()->for($pipeline)->for($pipeline->stages->get(1))->create();
        $stage = $pipeline->stages->get(0);

        $this->runAction($this->action, $board, ['stage_id' => $stage->id])->assertActionOk();
        $this->assertEquals($stage->id, $board->fresh()->stage_id);
    }

    public function test_authorized_user_can_run_change_board_stage_action(): void
    {
        $this->asRegularUser()->withPermissionsTo('edit all board')->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $board = $this->factory()->for($pipeline)->for($pipeline->stages->get(1))->for(User::factory())->create();
        $stage = $pipeline->stages->get(0);

        $this->runAction($this->action, $board, ['stage_id' => $stage->id])->assertActionOk();
        $this->assertEquals($stage->id, $board->fresh()->stage_id);
    }

    public function test_unauthorized_user_can_run_change_board_stage_action_on_own_board(): void
    {
        $signedInUser = $this->asRegularUser()->withPermissionsTo('edit own board')->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $boardForSignedIn = $this->factory()->for($pipeline)->for($pipeline->stages->get(1))->for($signedInUser)->create();
        $stage = $pipeline->stages->get(0);
        $otherBoard = $this->factory()->create();

        $this->runAction($this->action, $otherBoard, ['stage_id' => $stage->id])->assertActionUnauthorized();
        $this->runAction($this->action, $boardForSignedIn, ['stage_id' => $stage->id])->assertActionOk();
        $this->assertEquals($stage->id, $boardForSignedIn->fresh()->stage_id);
    }

    public function test_change_board_stage_action_requires_stage(): void
    {
        $this->signIn();

        $board = $this->factory()->create();

        $this->runAction($this->action, $board, ['stage_id' => ''])->assertJsonValidationErrors('stage_id');
    }

    public function test_it_updates_the_pipeline_id_if_the_provided_stage_does_not_belongs_to_the_current_board_pipeline(): void
    {
        $this->signIn();

        $board = $this->factory()->create();
        $stage = Stage::factory()->create();

        $this->runAction($this->action, $board, ['stage_id' => $stage->id])->assertActionOk();
        $this->assertEquals($board->fresh()->pipeline_id, $stage->pipeline_id);
    }
}
