<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Illuminate\Support\Facades\Event;
use Illuminate\Testing\Fluent\AssertableJson;
use Modules\Board\Board\Board;
use Modules\Board\Events\BoardMovedToStage;
use Modules\Board\Models\Board;
use Modules\Board\Models\Pipeline;
use Modules\Board\Models\Stage;
use Tests\TestCase;

class BoardBoardControllerTest extends TestCase
{
    public function test_unauthenticated_user_cannot_access_board_board_endpoints(): void
    {
        $pipeline = Pipeline::factory()->create();

        $this->getJson('/api/board/board')->assertUnauthorized();
        $this->getJson('/api/board/board/'.$pipeline->id.'/summary')->assertUnauthorized();
        $this->postJson('/api/board/board/'.$pipeline->id)->assertUnauthorized();
        $this->getJson('/api/board/board/'.$pipeline->id.'/FAKE_STAGE_ID')->assertUnauthorized();
    }

    public function test_board_can_be_updated_via_board(): void
    {
        $user = $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $board = Board::factory(3)->for($user)->for($pipeline)->create();
        $newStage = Stage::factory()->for($pipeline)->create();
        $boardOrder = [1000, 2000, 3000];

        Event::fake();

        $payload = $board->map(function ($board, $index) use ($newStage, $boardOrder) {
            return [
                'id' => $board->id,
                'board_order' => $boardOrder[$index],
                'stage_id' => $index === 2 ? $board->stage_id : $newStage->id,
                'swatch_color' => null,
            ];
        })->all();

        $this->postJson('/api/board/board/'.$pipeline->id, $payload)->assertOk();

        $board = $board->map->fresh();

        $this->assertEquals($board->get(0)->board_order, $boardOrder[0]);
        $this->assertEquals($board->get(1)->board_order, $boardOrder[1]);
        $this->assertEquals($board->get(2)->board_order, $boardOrder[2]);

        $this->assertEquals($board->get(0)->stage_id, $newStage->id);
        $this->assertEquals($board->get(1)->stage_id, $newStage->id);
        $this->assertEquals($board->get(2)->stage_id, $board->get(2)->stage_id);

        Event::assertDispatchedTimes(BoardMovedToStage::class, 2);
    }

    public function test_user_cannot_update_board_via_board_that_is_not_authorized_to_update(): void
    {
        $signedInUser = $this->asRegularUser()
            ->withPermissionsTo('edit own board')
            ->signIn();

        $user = $this->createUser();
        $pipeline = Pipeline::factory()->withStages()->create();
        $otherBoard = Board::factory()->for($pipeline)->for($user)->create(['board_order' => 500]);
        $boardForSignedIn = Board::factory()->for($pipeline)->for($signedInUser)->create(['board_order' => 501]);
        $newStage = Stage::factory()->for($pipeline)->create();

        Event::fake();

        $this->postJson('/api/board/board/'.$pipeline->id, [
            [
                'id' => $otherBoard->id,
                'board_order' => 1000,
                'stage_id' => $newStage->id,
                'swatch_color' => '#ffffff',
            ],
            [
                'id' => $boardForSignedIn->id,
                'board_order' => 1500,
                'stage_id' => $newStage->id,
                'swatch_color' => '#333333',
            ],
        ]);

        $this->assertEquals($otherBoard->fresh()->board_order, 500);
        $this->assertEquals($otherBoard->stage_id, $otherBoard->fresh()->stage_id);

        $this->assertEquals($boardForSignedIn->fresh()->board_order, 1500);
        $this->assertEquals($newStage->id, $boardForSignedIn->fresh()->stage_id);
        $this->assertEquals('#333333', $boardForSignedIn->fresh()->swatch_color);

        Event::assertDispatchedTimes(BoardMovedToStage::class, 1);
    }

    public function test_stage_moved_activity_is_logged_when_board_stage_is_updated_via_board(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();

        $board = Board::factory()->create();
        $newStage = Stage::factory()->for($pipeline)->create();

        $this->postJson('/api/board/board/'.$pipeline->id, [
            [
                'id' => $board->id,
                'stage_id' => $newStage->id,
                'board_order' => 1,
                'swatch_color' => null,
            ],
        ]);

        $latestActivity = $board->changeLog()->orderBy('id', 'desc')->first();

        $this->assertStringContainsString('board::board.timeline.stage.moved', (string) $latestActivity->properties);
    }

    public function test_triggers_model_events_when_updating_board_via_board(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $board = Board::factory(2)->create();

        Event::fake();

        $this->postJson('/api/board/board/'.$pipeline->id, [
            [
                'id' => $board->all()[0]->id,
                'stage_id' => $pipeline->stages[0]->id,
                'board_order' => 1,
                'swatch_color' => null,
            ],
            [
                'id' => $board->all()[1]->id,
                'stage_id' => $pipeline->stages[2]->id,
                'board_order' => 2,
                'swatch_color' => null,
            ],
        ]);

        // Assert an event was dispatched twice...
        Event::assertDispatched('eloquent.updating: '.Board::class, 2);
        Event::assertDispatched('eloquent.updated: '.Board::class, 2);
    }

    public function test_updated_at_column_is_updated_only_when_board_stage_changes(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $board = Board::factory(2)->create([
            'updated_at' => now()->subDays(2),
        ]);

        $updatedAt = $board[0]->updated_at->format('Y-m-d H:i:s');

        $this->postJson('/api/board/board/'.$pipeline->id, [
            [
                'id' => $board[0]->id,
                'stage_id' => $pipeline->stages->filter(
                    fn ($stage) => $stage->id !== $board[0]->stage_id
                )[0]->id,
                'board_order' => 2,
                'swatch_color' => null,
            ],
            [
                'id' => $board[1]->id,
                'stage_id' => $board[1]->stage_id,
                'board_order' => 3,
                'swatch_color' => null,
            ],
        ])->assertOk();

        $this->assertNotSame(
            $updatedAt,
            $board[0]->fresh()->updated_at->format('Y-m-d H:i:s')
        );

        $this->assertSame(
            $board[1]->updated_at->format('Y-m-d H:i:s'),
            $board[1]->fresh()->updated_at->format('Y-m-d H:i:s')
        );
    }

    /**
     * The board name is needed for the task create to add the name into the select
     */
    public function test_board_name_is_returned_in_board_json_resource(): void
    {
        $this->signIn();
        $pipeline = Pipeline::factory()->withStages()->create();
        Board::factory()->for($pipeline->stages->first())->create(['name' => 'Board Name']);

        $this->getJson('/api/board/board/'.$pipeline->id)
            ->assertJson(function (AssertableJson $json) {
                $json->has('0.cards.0', function ($json) {
                    $json->has('name')->where('name', 'Board Name')->etc();
                })->etc();
            });
    }

    public function test_all_pipeline_stages_are_returned_on_board_board(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        Board::factory()->for($pipeline)->create();

        $this->getJson('/api/board/board/'.$pipeline->id)->assertJsonCount(
            $pipeline->stages->count()
        );
    }

    public function test_board_board_filters_are_applied(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        $board = Board::factory(5)->for($pipeline)->create();

        $response = $this->getJson('/api/board/board/'.$pipeline->id.'?'.http_build_query([
            'filters' => [
                'condition' => 'and',
                'children' => [
                    [
                        'type' => 'rule',
                        'query' => [
                            'type' => 'text',
                            'rule' => 'name',
                            'operator' => 'equal',
                            'operand' => '',
                            'value' => $board->first()->name,
                        ],
                    ],
                ],
            ],
        ]));

        foreach ($response->getData() as $stage) {
            if ($stage->id === $board->first()->stage_id) {
                $this->assertCount(1, $stage->cards);
            } else {
                $this->assertCount(0, $stage->cards);
            }
        }
    }

    public function test_board_stages_are_in_correct_order(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();

        $last = $pipeline->stages->first();
        $last->display_order = 1000;
        $last->save();

        $first = $pipeline->stages->last();
        $first->display_order = 1;
        $first->save();

        $response = $this->getJson('/api/board/board/'.$pipeline->id);

        $lastIndex = $pipeline->stages->count() - 1;
        $this->assertEquals($response->getData()[0]->id, $first->id);
        $this->assertEquals($response->getData()[$lastIndex]->id, $last->id);
    }

    public function test_board_board_summary_can_be_retrieved(): void
    {
        $this->signIn();

        $pipeline = Pipeline::factory()->withStages()->create();
        Board::factory(5)->for($pipeline)->create();

        $response = $this->getJson('/api/board/board/'.$pipeline->id.'/summary')
            ->assertJsonCount($pipeline->stages->count());

        foreach ($response->getData() as $stageId => $summary) {
            $this->assertArrayHasKey('value', (array) $summary);
            $this->assertArrayHasKey('count', (array) $summary);
        }
    }

    public function test_it_loads_more_board(): void
    {
        $this->signIn();

        Board::$perPage = 5;
        $pipeline = Pipeline::factory()->has(Stage::factory())->create();

        Board::factory(10)->for($pipeline)->for($pipeline->stages[0])->create();
        $stage = $pipeline->stages[0];

        $this->getJson("/api/board/board/$pipeline->id/$stage->id?page=2")->assertJsonCount(5, 'cards');
    }

    public function test_it_can_fully_refresh_board_stage(): void
    {
        $this->signIn();

        Board::$perPage = 5;

        $pipeline = Pipeline::factory()->has(Stage::factory())->create();
        Board::factory(11)->for($pipeline)->for($pipeline->stages[0])->create();
        $stage = $pipeline->stages[0];

        $this->getJson("/api/board/board/$pipeline->id?pages[$stage->id]=2")->assertJsonCount(10, '0.cards');
    }
}
