<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Modules\Core\Tests\ResourceTestCase;
use Modules\Users\Models\User;

class BoardDeleteActionTest extends ResourceTestCase
{
    protected $resourceName = 'board';

    public function test_board_delete_action(): void
    {
        $this->signIn();

        $board = $this->factory()->count(2)->create();

        $this->runAction('delete', $board)->assertActionOk();
        $this->assertSoftDeleted('board', ['id' => $board->modelKeys()]);
    }

    public function test_unauthorized_user_cant_run_board_delete_action(): void
    {
        $this->asRegularUser()->signIn();

        $board = $this->factory()->for(User::factory())->count(2)->create();

        $this->runAction('delete', $board)->assertActionUnauthorized();
        $this->assertDatabaseHas('board', ['id' => $board->modelKeys()]);
    }

    public function test_authorized_user_can_run_board_delete_action(): void
    {
        $this->asRegularUser()->withPermissionsTo('delete any board')->signIn();

        $board = $this->factory()->for(User::factory())->create();

        $this->runAction('delete', $board)->assertActionOk();
        $this->assertSoftDeleted('board', ['id' => $board->id]);
    }

    public function test_authorized_user_can_run_board_delete_action_only_on_own_board(): void
    {
        $signedInUser = $this->asRegularUser()->withPermissionsTo('delete own board')->signIn();

        $boardForSignedIn = $this->factory()->for($signedInUser)->create();
        $otherboard = $this->factory()->create();

        $this->runAction('delete', $otherboard)->assertActionUnauthorized();
        $this->assertDatabaseHas('board', ['id' => $otherboard->id]);

        $this->runAction('delete', $boardForSignedIn);
        $this->assertSoftDeleted('board', ['id' => $boardForSignedIn->id]);
    }

    public function test_authorized_user_can_bulk_delete_board(): void
    {
        $this->asRegularUser()->withPermissionsTo([
            'delete any board', 'bulk delete board',
        ])->signIn();

        $board = $this->factory()->for(User::factory())->count(2)->create();

        $this->runAction('delete', $board);
        $this->assertSoftDeleted('board', ['id' => $board->modelKeys()]);
    }

    public function test_authorized_user_can_bulk_delete_only_own_board(): void
    {
        $signedInUser = $this->asRegularUser()->withPermissionsTo([
            'delete own board',
            'bulk delete board',
        ])->signIn();

        $boardForSignedInUser = $this->factory()->count(2)->for($signedInUser)->create();
        $otherboard = $this->factory()->create();

        $this->runAction('delete', $boardForSignedInUser->push($otherboard))->assertActionOk();
        $this->assertDatabaseHas('board', ['id' => $otherboard->id]);
        $this->assertSoftDeleted('board', ['id' => $boardForSignedInUser->modelKeys()]);
    }

    public function test_unauthorized_user_cant_bulk_delete_board(): void
    {
        $this->asRegularUser()->signIn();

        $board = $this->factory()->count(2)->create();

        $this->runAction('delete', $board)->assertActionUnauthorized();
        $this->assertDatabaseHas('board', ['id' => $board->modelKeys()]);
    }

    public function test_user_without_bulk_delete_permission_cannot_bulk_delete_board(): void
    {
        $this->asRegularUser()->withPermissionsTo([
            'delete any board',
            'delete own board',
            'delete team board',
        ])->signIn();

        $board = $this->factory()->for(User::factory())->count(2)->create();

        $this->runAction('delete', $board)->assertActionUnauthorized();
        $this->assertDatabaseHas('board', ['id' => $board->modelKeys()]);
    }
}
