<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Support\Carbon;
use Modules\Activities\Models\Activity;
use Modules\Calls\Models\Call;
use Modules\Contacts\Models\Company;
use Modules\Contacts\Models\Contact;
use Modules\Board\Enums\BoardStatus;
use Modules\Board\Models\Board;
use Modules\Board\Models\Pipeline;
use Modules\Board\Models\Stage;
use Modules\Notes\Models\Note;
use Modules\Users\Models\User;
use Tests\TestCase;

class BoardModelTest extends TestCase
{
    public function test_when_board_created_by_not_provided_uses_current_user_id(): void
    {
        $user = $this->signIn();

        $board = Board::factory(['created_by' => null])->create();

        $this->assertEquals($board->created_by, $user->id);
    }

    public function test_board_created_by_can_be_provided(): void
    {
        $user = $this->createUser();

        $board = Board::factory()->for($user, 'creator')->create();

        $this->assertEquals($board->created_by, $user->id);
    }

    public function test_board_has_user(): void
    {
        $board = Board::factory()->for(User::factory())->create();

        $this->assertInstanceOf(User::class, $board->user);
    }

    public function test_board_has_companies(): void
    {
        $board = Board::factory()->has(Company::factory()->count(2))->create();

        $this->assertCount(2, $board->companies);
    }

    public function test_board_has_contacts(): void
    {
        $board = Board::factory()->has(Contact::factory()->count(2))->create();

        $this->assertCount(2, $board->contacts);
    }

    public function test_board_has_calls(): void
    {
        $board = Board::factory()->has(Call::factory()->count(2))->create();

        $this->assertCount(2, $board->calls);
    }

    public function test_board_has_notes(): void
    {
        $board = Board::factory()->has(Note::factory()->count(2))->create();

        $this->assertCount(2, $board->notes);
    }

    public function test_board_has_activities(): void
    {
        $board = Board::factory()->has(Activity::factory()->count(2))->create();

        $this->assertCount(2, $board->activities);
    }

    public function test_board_has_pipeline(): void
    {
        $board = Board::factory()->for(Pipeline::factory()->withStages())->create();

        $this->assertInstanceOf(Pipeline::class, $board->pipeline);
    }

    public function test_board_has_stage(): void
    {
        $board = Board::factory()->for(Stage::factory())->create();

        $this->assertInstanceOf(Stage::class, $board->stage);
    }

    public function test_board_has_stages_history(): void
    {
        $board = new Board;

        $this->assertInstanceOf(BelongsToMany::class, $board->stagesHistory());
    }

    public function test_it_can_record_stage_history(): void
    {
        $board = Board::factory()->for(Stage::factory())->create();

        $board->recordStageHistory($board->stage_id);

        // With the one when the board is created
        $this->assertCount(2, $board->stagesHistory);
        $this->assertNotNull($board->lastStageHistory()['history']->entered_at);
        $this->assertEquals($board->lastStageHistory()->id, $board->stage_id);
    }

    public function test_it_can_start_board_stage_history(): void
    {
        $board = Board::factory()->for(Stage::factory())->create();

        $board->startStageHistory();

        // With the one when the board is created
        $this->assertCount(2, $board->stagesHistory);
        $this->assertNotNull($board->lastStageHistory()['history']->entered_at);
        $this->assertEquals($board->lastStageHistory()->id, $board->stage_id);
    }

    public function test_board_last_stage_history_can_be_retrieved(): void
    {
        $board = Board::factory()->for(Stage::factory())->open()->create();

        $this->assertInstanceOf(Stage::class, $board->lastStageHistory());
    }

    public function test_board_stage_history_is_started_when_open_board_is_created(): void
    {
        $board = Board::factory()->for(Stage::factory())->open()->create();

        $this->assertNotEmpty($board->stagesHistory);
        $this->assertCount(1, $board->stagesHistory);
        $this->assertNotNull($board->stagesHistory[0]['history']->entered_at);
        $this->assertNull($board->stagesHistory[0]['history']->left_at);
    }

    public function test_board_stage_history_is_not_started_when_won_board_is_created(): void
    {
        $board = Board::factory()->for(Stage::factory())->won()->create();

        $this->assertEmpty($board->stagesHistory);
        $this->assertCount(0, $board->stagesHistory);
    }

    public function test_board_stage_history_is_not_started_when_lost_board_is_created(): void
    {
        $board = Board::factory()->for(Stage::factory())->lost()->create();

        $this->assertEmpty($board->stagesHistory);
        $this->assertCount(0, $board->stagesHistory);
    }

    public function test_board_stage_history_is_stopped_when_status_is_changed_to_won(): void
    {
        $board = Board::factory()->for(Stage::factory())->open()->create();
        $board->markAsWon();

        $this->assertNotNull($board->lastStageHistory()['history']->left_at);
    }

    public function test_board_stage_history_is_stopped_when_status_is_changed_to_lost(): void
    {
        $board = Board::factory()->for(Stage::factory())->open()->create();
        $board->markAsLost();

        $this->assertNotNull($board->lastStageHistory()['history']->left_at);
    }

    public function test_board_last_stage_history_timing_can_be_stopped(): void
    {
        $board = Board::factory()->for(Stage::factory())->open()->create();

        $board->stopLastStageTiming();

        $this->assertNotNull($board->stagesHistory[0]['history']->left_at);
    }

    public function test_board_history_stages_are_always_sorted_by_newest(): void
    {
        $enteredAt = '2021-11-21 12:00:00';
        Carbon::setTestNow($enteredAt);
        $board = Board::factory()->for(Stage::factory())->create();
        Carbon::setTestNow(null);
        $board->stopLastStageTiming();
        $enteredAt = '2021-11-21 12:05:00';
        Carbon::setTestNow($enteredAt);
        $board->startStageHistory();
        $this->assertEquals($enteredAt, $board->stagesHistory[0]['history']->entered_at);
    }

    public function test_board_time_in_stages_is_properly_calculated(): void
    {
        $stages = Stage::factory()->count(2)->create();

        $enteredAtForStage1 = '2021-11-21 12:00:00';
        Carbon::setTestNow($enteredAtForStage1);
        $board = Board::factory()->for($stages[0])->create();
        $leftAtForStage1 = '2021-11-21 12:05:00';

        Carbon::setTestNow($leftAtForStage1);
        $board->stopLastStageTiming(); // total time 5 minutes, 300 in seconds

        $board->stage_id = $stages[1]->id;
        $enteredAtForStage2 = '2021-11-21 12:06:00';
        Carbon::setTestNow($enteredAtForStage2);
        $board->save();

        $leftAtForStage2 = '2021-11-21 12:10:00';
        Carbon::setTestNow($leftAtForStage2);
        $board->stopLastStageTiming(); // total time 4 minutes, 240 in seconds

        $timeInStages = $board->timeInStages();

        $this->assertEquals(300, $timeInStages[$stages[0]->id]);
        $this->assertEquals(240, $timeInStages[$stages[1]->id]);
    }

    public function test_board_stage_changed_date_is_updated_when_stage_is_changed(): void
    {
        $stages = Stage::factory()->count(2)->create();
        $board = Board::factory()->for($stages[0])->create();

        $board->stage_id = $stages[1]->id;
        $board->save();

        $this->assertNotNull($board->stage_changed_date);
    }

    public function test_stage_history_is_started_when_stage_is_changed_and_board_is_with_status_open(): void
    {
        $stages = Stage::factory()->count(2)->create();
        $board = Board::factory()->for($stages[0])->open()->create();

        $board->stage_id = $stages[1]->id;
        $board->save();

        // +1 from stage history when created
        $this->assertCount(2, $board->stagesHistory);
    }

    public function test_stage_history_is_not_started_when_stage_is_changed_and_board_is_with_status_won(): void
    {
        $stages = Stage::factory()->count(2)->create();
        $board = Board::factory()->for($stages[0])->won()->create();

        $board->stage_id = $stages[1]->id;
        $board->save();

        $this->assertCount(0, $board->stagesHistory);
    }

    public function test_stage_history_is_not_started_when_stage_is_changed_and_board_is_with_status_lost(): void
    {
        $stages = Stage::factory()->count(2)->create();
        $board = Board::factory()->for($stages[0])->lost()->create();

        $board->stage_id = $stages[1]->id;
        $board->save();

        $this->assertCount(0, $board->stagesHistory);
    }

    public function test_it_does_not_stop_latest_stage_history_if_its_already_stopped(): void
    {
        $board = Board::factory()->for(Stage::factory())->create();

        $board->stopLastStageTiming();
        $lastStoppedTime = $board->stagesHistory[0]['history']->left_at;
        $board->stopLastStageTiming();

        $this->assertEquals($lastStoppedTime, $board->stagesHistory()->first()['history']->left_at);
    }

    public function test_board_status_attributes_are_properly_updated(): void
    {
        // Create
        $board1 = Board::factory()->open()->create();

        $this->assertNull($board1->won_date);
        $this->assertNull($board1->lost_date);
        $this->assertNull($board1->lost_reason);

        $board2 = Board::factory()->won()->create(['lost_reason' => 'Reason', 'lost_date' => now()]);

        $this->assertNotNull($board2->won_date);
        $this->assertNull($board2->lost_date);
        $this->assertNull($board2->lost_reason);

        $board3 = Board::factory()->lost()->create(['won_date' => now()]);

        $this->assertNotNull($board3->lost_date);
        $this->assertNull($board3->won_date);
        $this->assertNull($board3->lost_reason);

        // Update
        $board3->markAsWon();

        $this->assertNotNull($board3->won_date);
        $this->assertNull($board3->lost_date);
        $this->assertNull($board3->lost_reason);

        $board2->markAsLost('Updated Lost Reason');

        $this->assertNull($board2->won_date);
        $this->assertNotNull($board2->lost_date);
        $this->assertEquals('Updated Lost Reason', $board2->lost_reason);
    }

    public function test_it_does_not_allow_changing_status_attributes_manually_when_updating(): void
    {
        $board = Board::factory()->open()->create();
        $board->won_date = '2021-11-21 12:00:00';
        $board->lost_date = '2021-11-21 12:00:00';
        $board->lost_reason = 'Lost Reason';
        $board->save();

        $this->assertNull($board->won_date);
        $this->assertNull($board->lost_date);
        $this->assertNull($board->lost_reason);
    }

    public function test_it_does_allow_changing_the_lost_reason_when_board_is_with_status_lost(): void
    {
        $board = Board::factory()->lost()->create();
        $board->lost_reason = 'Changed Reason';
        $board->save();
        $this->assertEquals('Changed Reason', $board->lost_reason);
    }

    public function test_stages_history_is_started_when_board_is_marked_as_open(): void
    {
        $board = Board::factory()->lost()->create();

        $board->markAsOpen();

        $this->assertCount(1, $board->stagesHistory);
    }

    public function test_board_has_badges_for_status(): void
    {
        $variants = BoardStatus::badgeVariants();

        $this->assertCount(3, $variants);
        $this->assertEquals('neutral', $variants['open']);
        $this->assertEquals('success', $variants['won']);
        $this->assertEquals('danger', $variants['lost']);
    }

    public function test_board_has_total_column(): void
    {
        $this->assertEquals('amount', (new Board)->totalColumn());
    }

    public function test_it_queries_closed_board(): void
    {
        Board::factory()->won()->create();
        Board::factory()->lost()->create();
        Board::factory()->open()->create();

        $this->assertSame(2, Board::closed()->count());
    }

    public function test_it_queries_won_board(): void
    {
        Board::factory()->won()->create();
        Board::factory()->lost()->create();
        Board::factory()->open()->create();

        $this->assertSame(1, Board::won()->count());
    }

    public function test_it_queries_open_board(): void
    {
        Board::factory()->won()->create();
        Board::factory()->lost()->create();
        Board::factory()->open()->create();

        $this->assertSame(1, Board::open()->count());
    }

    public function test_it_queries_lost_board(): void
    {
        Board::factory()->won()->create();
        Board::factory()->lost()->create();
        Board::factory()->open()->create();

        $this->assertSame(1, Board::lost()->count());
    }
}
