<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Event;
use Modules\Activities\Models\Activity;
use Modules\Billable\Models\Billable;
use Modules\Billable\Models\BillableProduct;
use Modules\Calls\Models\Call;
use Modules\Contacts\Models\Company;
use Modules\Contacts\Models\Contact;
use Modules\Core\Http\Requests\ResourceRequest;
use Modules\Core\Models\ModelVisibilityGroup;
use Modules\Core\Tests\ResourceTestCase;
use Modules\Board\Enums\BoardStatus;
use Modules\Board\Events\BoardMovedToStage;
use Modules\Board\Models\Board;
use Modules\Board\Models\Pipeline;
use Modules\Board\Models\Stage;
use Modules\Notes\Models\Note;
use Modules\Users\Models\Team;
use Modules\Users\Models\User;

class BoardResourceTest extends ResourceTestCase
{
    protected $resourceName = 'board';

    public function test_user_can_create_board(): void
    {
        $this->signIn();
        $user = $this->createUser();
        $pipeline = Pipeline::factory()->withStages()->create();
        $stage = $pipeline->stages->first();
        $company = Company::factory()->create();
        $contact = Contact::factory()->create();

        $response = $this->postJson($this->createEndpoint(), [
            'name' => 'Board Name',
            'expected_close_date' => $closeDate = now()->addMonth()->format('Y-m-d'),
            'pipeline_id' => $pipeline->id,
            'amount' => 1250,
            'stage_id' => $stage->id,
            'user_id' => $user->id,
            'companies' => [$company->id],
            'contacts' => [$contact->id],
        ])
            ->assertCreated();

        $this->assertResourceJsonStructure($response);
        $response->assertJsonCount(1, 'companies')
            ->assertJsonCount(1, 'contacts')
            ->assertJson([
                'companies' => [['id' => $company->id]],
                'contacts' => [['id' => $contact->id]],
                'name' => 'Board Name',
                'expected_close_date' => Carbon::parse($closeDate)->toJSON(),
                'pipeline_id' => (string) $pipeline->id,
                'amount' => (string) 1250,
                'stage_id' => (string) $stage->id,
                'user_id' => (string) $user->id,
                'was_recently_created' => true,
                'display_name' => 'Board Name',
                'companies_count' => 1,
                'contacts_count' => 1,
            ]);
    }

    public function test_user_can_update_board(): void
    {
        $user = $this->signIn();
        $pipeline = Pipeline::factory()->withStages()->create();
        $stage = $pipeline->stages->first();
        $company = Company::factory()->create();
        $contact = Contact::factory()->create();
        $record = $this->factory()->has(Company::factory())->create();

        $response = $this->putJson($this->updateEndpoint($record), [
            'name' => 'Board Name',
            'expected_close_date' => $closeDate = now()->addMonth()->format('Y-m-d'),
            'pipeline_id' => $pipeline->id,
            'amount' => 3655,
            'stage_id' => $stage->id,
            'user_id' => $user->id,
            'companies' => [$company->id],
            'contacts' => [$contact->id],
        ])
            ->assertOk();

        $this->assertResourceJsonStructure(($response));

        $response->assertJsonCount(count($this->resource()->resolveActions(app(ResourceRequest::class))), 'actions')
            ->assertJsonCount(1, 'companies')
            ->assertJsonCount(1, 'contacts')
            ->assertJson([
                'companies' => [['id' => $company->id]],
                'contacts' => [['id' => $contact->id]],
                'name' => 'Board Name',
                'expected_close_date' => Carbon::parse($closeDate)->toJSON(),
                'pipeline_id' => (string) $pipeline->id,
                'amount' => (string) 3655,
                'stage_id' => (string) $stage->id,
                'user_id' => (string) $user->id,
                'display_name' => 'Board Name',
                'companies_count' => 1,
                'contacts_count' => 1,
            ]);
    }

    public function test_user_can_retrieve_board(): void
    {
        $this->signIn();

        $this->factory()->count(5)->create();

        $this->getJson($this->indexEndpoint())->assertJsonCount(5, 'data');
    }

    public function test_user_can_retrieve_board(): void
    {
        $this->signIn();

        $record = $this->factory()->create();

        $this->getJson($this->showEndpoint($record))->assertOk();
    }

    public function test_user_can_globally_search_board(): void
    {
        $this->signIn();

        $record = $this->factory()->create();

        $this->getJson("/api/search?q={$record->name}&only=board")
            ->assertJsonCount(1, '0.data')
            ->assertJsonPath('0.data.0.id', $record->id)
            ->assertJsonPath('0.data.0.path', "/board/{$record->id}")
            ->assertJsonPath('0.data.0.display_name', $record->name);
    }

    public function test_an_unauthorized_user_can_global_search_only_board(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('view own board')->signIn();
        $user1 = $this->createUser();

        $this->factory()->for($user1)->create(['name' => 'DEAL KONKORD']);
        $record = $this->factory()->for($user)->create(['name' => 'DEAL INOKLAPS']);

        $this->getJson('/api/search?q=DEAL&only=board')
            ->assertJsonCount(1, '0.data')
            ->assertJsonPath('0.data.0.id', $record->id)
            ->assertJsonPath('0.data.0.path', "/board/{$record->id}")
            ->assertJsonPath('0.data.0.display_name', $record->name);
    }

    public function test_user_can_export_board(): void
    {
        $this->performExportTest();
    }

    public function test_user_can_create_board_with_custom_fields(): void
    {
        $this->signIn();

        $response = $this->postJson(
            $this->createEndpoint(),
            array_merge($this->samplePayload(), $this->customFieldsPayload())
        )->assertCreated();

        $this->assertThatResponseHasCustomFieldsValues($response);
    }

    public function test_user_can_update_board_with_custom_fields(): void
    {
        $this->signIn();
        $record = $this->factory()->create();

        $response = $this->putJson(
            $this->updateEndpoint($record),
            array_merge($this->samplePayload(), $this->customFieldsPayload())
        )->assertOk();

        $this->assertThatResponseHasCustomFieldsValues($response);
    }

    public function test_user_can_import_board(): void
    {
        $this->signIn();

        $this->performImportTest();
    }

    public function test_user_can_import_board_with_custom_fields(): void
    {
        $this->signIn();

        $this->performImportWithCustomFieldsTest();
    }

    protected function performImportTest($overrides = []): void
    {
        Pipeline::factory()->withStages()->create();
        parent::performExportTest($overrides);
    }

    protected function performImportWithCustomFieldsTest(): void
    {
        Pipeline::factory()->withStages()->create();
        parent::performImportWithCustomFieldsTest();
    }

    protected function importEndpoint($import): string
    {
        $id = is_int($import) ? $import : $import->getKey();
        $pipeline = Pipeline::first();

        return "/api/{$this->resourceName}/import/{$id}?pipeline_id={$pipeline->id}";
    }

    public function test_user_can_load_the_board_table(): void
    {
        $this->performTestTableLoad();
    }

    public function test_board_table_loads_all_fields(): void
    {
        $this->performTestTableCanLoadWithAllFields();
    }

    public function test_it_doesnt_create_board_with_restricted_visibility_pipeline(): void
    {
        $this->asRegularUser()->signIn();

        $pipeline = $this->newPipelineFactoryWithVisibilityGroup('users', User::factory())->create();

        $this->postJson(
            $this->createEndpoint(),
            ['pipeline_id' => $pipeline->id]
        )
            ->assertJsonValidationErrors(['pipeline_id' => 'This Pipeline value is forbidden.']);
    }

    public function test_it_doesnt_update_board_with_restricted_visibility_pipeline(): void
    {
        $this->asRegularUser()->withPermissionsTo('edit all board')->signIn();
        $board = $this->factory()->create();
        $pipeline = $this->newPipelineFactoryWithVisibilityGroup('users', User::factory())->create();

        $this->putJson(
            $this->updateEndpoint($board),
            ['pipeline_id' => $pipeline->id]
        )->assertJsonValidationErrors(['pipeline_id' => 'This Pipeline value is forbidden.']);
    }

    public function test_stage_id_is_required_when_pipeline_is_provided(): void
    {
        $this->signIn();
        $board = $this->factory()->create();

        $this->putJson(
            $this->updateEndpoint($board),
            [
                'pipeline_id' => $board->pipeline_id,
                'name' => 'Changed Name',
            ]
        )
            ->assertJsonValidationErrorFor('stage_id');
    }

    public function test_user_can_update_board_with_same_restricted_visibility_pipeline(): void
    {
        $this->asRegularUser()->withPermissionsTo(['view all board', 'edit all board'])->signIn();
        $pipeline = $this->newPipelineFactoryWithVisibilityGroup('users', User::factory())->withStages()->create();
        $board = $this->factory()->for($pipeline)->create();

        $this->putJson(
            $this->updateEndpoint($board),
            [
                'pipeline_id' => $pipeline->id,
                'stage_id' => $board->stage_id,
                'name' => 'Changed Name',
            ]
        )
            ->assertOk()
            ->assertJson([
                'pipeline_id' => $pipeline->id,
                'name' => 'Changed Name',
            ]);
    }

    public function test_it_updates_stage_when_the_pipeline_is_restricted(): void
    {
        $this->asRegularUser()->withPermissionsTo(['edit all board'])->signIn();
        $pipeline = $this->newPipelineFactoryWithVisibilityGroup('users', User::factory())->withStages()->create();
        $board = $this->factory()->for($pipeline)->create();
        $stage = Stage::factory()->for($pipeline)->create();

        $this->putJson($this->updateEndpoint($board), ['stage_id' => $stage->id])
            ->assertOk()
            ->assertJson(['stage_id' => $stage->id]);
    }

    public function test_when_creating_it_uses_stage_pipeline_when_pipeline_is_not_provided(): void
    {
        $this->signIn();

        $stage = Stage::factory()->create();

        $this->postJson(
            $this->createEndpoint(),
            [
                'name' => 'Board Name',
                'stage_id' => $stage->id,
            ]
        );

        $board = Board::first();

        $this->assertEquals($board->stage->pipeline_id, $board->pipeline_id);
    }

    public function test_board_is_by_default_open(): void
    {
        $this->signIn();

        $stage = Stage::factory()->create();

        $this->postJson(
            $this->createEndpoint(),
            [
                'name' => 'Board Name',
                'stage_id' => $stage->id,
            ]
        )->assertJson(['status' => BoardStatus::open->name]);
    }

    public function test_status_can_be_provided_when_creating_new_board(): void
    {
        $this->signIn();

        $stage = Stage::factory()->create();

        $this->postJson(
            $this->createEndpoint(),
            [
                'name' => 'Board Name',
                'stage_id' => $stage->id,
                'status' => BoardStatus::lost->name,
            ]
        )->assertJson(['status' => BoardStatus::lost->name]);
    }

    public function test_status_can_be_provided_when_updating_a_board(): void
    {
        $this->signIn();

        $board = Board::factory()->open()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
            'lost_reason' => 'Lost Reason',
        ])->assertJson(['status' => BoardStatus::lost->name, 'lost_reason' => 'Lost Reason']);
    }

    public function test_it_starts_stage_history_when_board_status_is_open(): void
    {
        $this->signIn();

        $stage = Stage::factory()->create();

        $this->postJson(
            $this->createEndpoint(),
            [
                'name' => 'Board Name',
                'stage_id' => $stage->id,
                'status' => BoardStatus::open->name,
            ]
        );

        $board = Board::first();

        $this->assertCount(1, $board->stagesHistory);

        $board->markAsLost();

        $this->putJson(
            $this->updateEndpoint($board),
            [
                'status' => BoardStatus::open->name,
            ]
        );

        $this->assertCount(2, $board->fresh()->stagesHistory);
    }

    public function test_it_requires_an_open_board_to_mark_as_lost(): void
    {
        $this->signIn();

        $board = Board::factory()->won()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
        ])->assertStatusConflict();
    }

    public function test_it_requires_an_open_board_to_mark_as_won(): void
    {
        $this->signIn();

        $board = Board::factory()->lost()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::won->name,
        ])->assertStatusConflict();
    }

    public function test_it_does_not_update_lost_reason_when_marking_board_as_won(): void
    {
        $this->signIn();

        $board = Board::factory()->open()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::won->name,
            'lost_reason' => 'Lost Reason',
        ])->assertJson(['lost_reason' => null]);
    }

    public function test_it_does_not_update_lost_reason_when_marking_board_as_open(): void
    {
        $this->signIn();

        $board = Board::factory()->won()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::open->name,
            'lost_reason' => 'Lost Reason',
        ])->assertJson(['lost_reason' => null]);
    }

    public function test_it_does_not_trigger_errors_when_closed_board_same_status_is_provided(): void
    {
        $this->signIn();

        $board = Board::factory()->lost()->create();

        $this->putJson($this->updateEndpoint($board), [
            'name' => 'Updated Name',
            'status' => BoardStatus::lost->name,
        ])->assertOk()->assertJson([
            'name' => 'Updated Name',
            'status' => BoardStatus::lost->name,
        ]);
    }

    public function test_lost_reason_can_be_updated_when_board_is_lost(): void
    {
        // api usage only
        $this->signIn();

        $board = Board::factory()->lost('Original Lost Reason')->create();

        $this->putJson($this->updateEndpoint($board), [
            'lost_reason' => 'Changed Lost Reason',
        ])->assertOk()->assertJson([
            'lost_reason' => 'Changed Lost Reason',
        ]);
    }

    public function test_lost_reason_can_be_optional(): void
    {
        $this->signIn();
        settings(['lost_reason_is_required' => false]);

        $board = Board::factory()->open()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
            'lost_reason' => '',
        ])->assertJsonMissingValidationErrors('lost_reason');

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
            'lost_reason' => null,
        ])->assertJsonMissingValidationErrors('lost_reason');

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
        ])->assertJsonMissingValidationErrors('lost_reason');
    }

    public function test_lost_reason_can_be_required(): void
    {
        $this->signIn();
        settings(['lost_reason_is_required' => true]);

        $board = Board::factory()->open()->create();

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
            'lost_reason' => '',
        ])->assertJsonValidationErrorFor('lost_reason');

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
            'lost_reason' => null,
        ])->assertJsonValidationErrorFor('lost_reason');

        $this->putJson($this->updateEndpoint($board), [
            'status' => BoardStatus::lost->name,
        ])->assertJsonValidationErrorFor('lost_reason');
    }

    public function test_it_does_not_update_only_pipeline(): void
    {
        $pipeline = Pipeline::factory()->has(Stage::factory())->create();

        $board = Board::factory()->create();

        $this->putJson($this->updateEndpoint($board), [
            'pipeline_id' => $pipeline->id,
        ]);

        $board->refresh();

        $this->assertNotEquals($pipeline->id, $board->pipeline_id);
    }

    public function test_when_updating_it_uses_stage_pipeline_when_pipeline_is_not_provided(): void
    {
        $pipeline = Pipeline::factory()->has(Stage::factory())->create();

        $board = Board::factory([
            'pipeline_id' => $pipeline->id,
            'stage_id' => $pipeline->stages[0]->id,
        ])->create();

        $this->putJson($this->updateEndpoint($board), [
            'pipeline_id' => null,
            'stage_id' => $board->stage_id,
        ]);

        $board->refresh();

        $this->assertEquals($board->stage->pipeline_id, $board->pipeline_id);
    }

    public function test_when_creating_it_uses_stage_pipeline_when_provided_pipeline_id_does_not_belong_to_the_stage(): void
    {
        $this->signIn();

        $otherPipeline = Pipeline::factory()->create();
        $mainPipeline = Pipeline::factory()->has(Stage::factory())->create();

        $this->postJson($this->createEndpoint(), [
            'name' => 'Board Name',
            'pipeline_id' => $otherPipeline->id,
            'stage_id' => $mainPipeline->stages[0]->id,
        ]);

        $board = Board::first();

        $this->assertEquals($board->stage->pipeline_id, $board->pipeline_id);
    }

    public function test_when_updating_it_uses_stage_pipeline_id_when_provided_pipeline_id_does_not_belong_to_the_stage(): void
    {
        $this->signIn();

        $otherPipeline = Pipeline::factory()->create();
        $board = Board::factory()->for(Pipeline::factory()->has(Stage::factory()))->create();

        $this->putJson($this->updateEndpoint($board), [
            'pipeline_id' => $otherPipeline->id,
            'stage_id' => $board->pipeline->stages[0]->id,
        ]);

        $board->refresh();

        $this->assertEquals($board->stage->pipeline_id, $board->pipeline_id);
    }

    public function test_moved_to_stage_event_is_triggered_when_board_stage_is_updated(): void
    {
        $this->signIn();

        $board = Board::factory()->create();
        $stageId = Stage::where('id', '!=', $board->stage_id)->first()->id;

        Event::fake();

        $this->putJson($this->updateEndpoint($board), [
            'stage_id' => $stageId,
        ]);

        Event::assertDispatched(BoardMovedToStage::class);
    }

    public function test_stage_moved_activity_is_logged_when_board_stage_is_updated(): void
    {
        $this->signIn();

        $board = Board::factory()->create();
        $stageId = Stage::where('id', '!=', $board->stage_id)->first()->id;

        $this->putJson($this->updateEndpoint($board), [
            'stage_id' => $stageId,
        ]);

        $latestActivity = $board->changelog()->orderBy('id', 'desc')->first();
        $this->assertStringContainsString('board::board.timeline.stage.moved', (string) $latestActivity->properties);
    }

    public function test_user_can_force_delete_board(): void
    {
        $this->signIn();

        $record = $this->factory()
            ->has(Contact::factory())
            ->has(Company::factory())
            ->has(Note::factory())
            ->has(Call::factory())
            ->has(Activity::factory())
            ->create();

        Billable::factory()
            ->withBillableable($record)
            ->has(BillableProduct::factory(), 'products')
            ->create();

        $record->delete();

        $this->deleteJson($this->forceDeleteEndpoint($record))->assertNoContent();
        $this->assertDatabaseCount($this->tableName(), 0);
        $this->assertDatabaseCount('billables', 0);
    }

    public function test_user_can_soft_delete_board(): void
    {
        $this->signIn();

        $record = $this->factory()->create();

        $this->deleteJson($this->deleteEndpoint($record))->assertNoContent();
        $this->assertDatabaseCount($this->tableName(), 1);
    }

    public function test_board_can_be_viewed_without_own_permissions(): void
    {
        $user = $this->asRegularUser()->signIn();
        $record = $this->factory()->for($user)->create();

        $this->getJson($this->showEndpoint($record))->assertOk()->assertJson(['id' => $record->id]);
    }

    public function test_edit_all_board_permission(): void
    {
        $this->asRegularUser()->withPermissionsTo('edit all board')->signIn();
        $record = $this->factory()->create();

        $this->putJson($this->updateEndpoint($record), $this->samplePayload())->assertOk();
    }

    public function test_edit_own_board_permission(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('edit own board')->signIn();
        $record1 = $this->factory()->for($user)->create();
        $record2 = $this->factory()->create();

        $payload = $this->samplePayload();
        $this->putJson($this->updateEndpoint($record1), $payload)->assertOk();
        $this->putJson($this->updateEndpoint($record2), $payload)->assertForbidden();
    }

    public function test_edit_team_board_permission(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('edit team board')->signIn();
        $teamUser = User::factory()->has(Team::factory()->for($user, 'manager'))->create();

        $record = $this->factory()->for($teamUser)->create();

        $this->putJson($this->updateEndpoint($record))->assertOk();
    }

    public function test_unauthorized_user_cannot_update_board(): void
    {
        $this->asRegularUser()->signIn();
        $record = $this->factory()->create();

        $this->putJson($this->updateEndpoint($record), $this->samplePayload())->assertForbidden();
    }

    public function test_view_all_board_permission(): void
    {
        $this->asRegularUser()->withPermissionsTo('view all board')->signIn();
        $record = $this->factory()->create();

        $this->getJson($this->showEndpoint($record))->assertOk();
    }

    public function test_view_team_board_permission(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('view team board')->signIn();
        $teamUser = User::factory()->has(Team::factory()->for($user, 'manager'))->create();

        $record = $this->factory()->for($teamUser)->create();

        $this->getJson($this->showEndpoint($record))->assertOk();
    }

    public function test_user_can_view_own_board(): void
    {
        $user = $this->asRegularUser()->signIn();
        $record = $this->factory()->for($user)->create();

        $this->getJson($this->showEndpoint($record))->assertOk();
    }

    public function test_unauthorized_user_cannot_view_board(): void
    {
        $this->asRegularUser()->signIn();
        $record = $this->factory()->create();

        $this->getJson($this->showEndpoint($record))->assertForbidden();
    }

    public function test_delete_any_board_permission(): void
    {
        $this->asRegularUser()->withPermissionsTo('delete any board')->signIn();

        $record = $this->factory()->create();

        $this->deleteJson($this->deleteEndpoint($record))->assertNoContent();
    }

    public function test_delete_own_board_permission(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('delete own board')->signIn();

        $record1 = $this->factory()->for($user)->create();
        $record2 = $this->factory()->create();

        $this->deleteJson($this->deleteEndpoint($record1))->assertNoContent();
        $this->deleteJson($this->deleteEndpoint($record2))->assertForbidden();
    }

    public function test_delete_team_board_permission(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo('delete team board')->signIn();
        $teamUser = User::factory()->has(Team::factory()->for($user, 'manager'))->create();

        $record1 = $this->factory()->for($teamUser)->create();
        $record2 = $this->factory()->create();

        $this->deleteJson($this->deleteEndpoint($record1))->assertNoContent();
        $this->deleteJson($this->deleteEndpoint($record2))->assertForbidden();
    }

    public function test_unauthorized_user_cannot_delete_board(): void
    {
        $this->asRegularUser()->signIn();
        $record = $this->factory()->create();

        $this->deleteJson($this->showEndpoint($record))->assertForbidden();
    }

    public function test_it_empties_board_trash(): void
    {
        $this->signIn();

        $this->factory()->count(2)->trashed()->create();

        $this->deleteJson('/api/trashed/board?limit=2')->assertJson(['deleted' => 2]);
        $this->assertDatabaseEmpty('board');
    }

    public function test_it_excludes_unauthorized_records_from_empty_board_trash(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo(['view own board', 'delete own board', 'bulk delete board'])->signIn();

        $this->factory()->trashed()->create();
        $this->factory()->trashed()->for($user)->create();

        $this->deleteJson('/api/trashed/board')->assertJson(['deleted' => 1]);
        $this->assertDatabaseCount('board', 1);
    }

    public function test_it_does_not_empty_board_trash_if_delete_own_board_permission_is_not_applied(): void
    {
        $user = $this->asRegularUser()->withPermissionsTo(['view own board', 'bulk delete board'])->signIn();

        $this->factory()->trashed()->for($user)->create();

        $this->deleteJson('/api/trashed/board')->assertJson(['deleted' => 0]);
        $this->assertDatabaseCount('board', 1);
    }

    public function test_board_has_view_route(): void
    {
        $model = $this->factory()->create();

        $this->assertEquals('/board/'.$model->id, $this->resource()->viewRouteFor($model));
    }

    public function test_board_has_title(): void
    {
        $model = $this->factory()->make(['name' => 'Board Name']);

        $this->assertEquals('Board Name', $this->resource()->titleFor($model));
    }

    protected function samplePayload()
    {
        $pipeline = Pipeline::factory()->withStages()->create();
        $stage = $pipeline->stages->first();

        return [
            'name' => 'Board Name',
            'expected_close_date' => now()->addMonth()->format('Y-m-d'),
            'pipeline_id' => $pipeline->id,
            'amount' => 1250,
            'stage_id' => $stage->id,
        ];
    }

    protected function newPipelineFactoryWithVisibilityGroup($group, $attached)
    {
        return Pipeline::factory()->has(
            ModelVisibilityGroup::factory()->{$group}()->hasAttached($attached),
            'visibilityGroup'
        );
    }

    protected function assertResourceJsonStructure($response)
    {
        $response->assertJsonStructure([
            'actions', 'amount', 'board_order', 'calls_count', 'companies', 'companies_count', 'contacts', 'contacts_count', 'created_at', 'display_name', 'expected_close_date', 'id', 'media', 'name', 'next_activity_date', 'notes_count', 'owner_assigned_date', 'pipeline', 'pipeline_id', 'stage', 'stage_changed_date', 'stage_id', 'status', 'time_in_stages', 'timeline_subject_key', 'incomplete_activities_for_user_count', 'unread_emails_for_user_count', 'updated_at', 'path', 'user', 'user_id', 'was_recently_created', 'tags', 'authorizations' => [
                'create', 'delete', 'update', 'view', 'viewAny',
            ],
        ]);

        if ($response->getData()->status == BoardStatus::won->name) {
            $response->assertResourceJsonStructure(['won_date']);
        }

        if ($response->getData()->status == BoardStatus::lost->name) {
            $response->assertResourceJsonStructure(['lost_date', 'lost_reason']);
        }
    }
}
