<?php
/**
 * Concord CRM - https://www.concordcrm.com
 *
 * @version   1.5.0
 *
 * @link      Releases - https://www.concordcrm.com/releases
 * @link      Terms Of Service - https://www.concordcrm.com/terms
 *
 * @copyright Copyright (c) 2022-2024 KONKORD DIGITAL
 */

namespace Modules\Board\Tests\Feature;

use Modules\Core\Tests\ResourceTestCase;
use Modules\Board\Enums\BoardStatus;
use Modules\Users\Models\User;

class MarkAsLostTest extends ResourceTestCase
{
    protected $action = 'mark-as-lost';

    protected $resourceName = 'board';

    public function test_super_admin_user_can_run_board_mark_as_lost_action(): void
    {
        $this->signIn();

        $board = $this->factory()->create();

        $this->runAction($this->action, $board)->assertActionOk();
        $this->assertSame(BoardStatus::lost, $board->fresh()->status);
    }

    public function test_lost_reason_can_be_provided(): void
    {
        $this->signIn();

        $board = $this->factory()->create();

        $this->runAction($this->action, $board, ['lost_reason' => 'Probably cause'])->assertActionOk();
        $this->assertSame(BoardStatus::lost, $board->fresh()->status);
        $this->assertSame('Probably cause', $board->fresh()->lost_reason);
    }

    public function test_lost_reason_can_be_required_provided(): void
    {
        $this->signIn();

        settings(['lost_reason_is_required' => true]);

        $board = $this->factory()->create();

        $this->runAction($this->action, $board, ['lost_reason' => ''])->assertJsonValidationErrorFor('lost_reason');
    }

    public function test_authorized_user_can_run_board_mark_as_lost_action(): void
    {
        $this->asRegularUser()->withPermissionsTo('edit all board')->signIn();

        $board = $this->factory()->for(User::factory())->create();

        $this->runAction($this->action, $board)->assertActionOk();
        $this->assertSame(BoardStatus::lost, $board->fresh()->status);
    }

    public function test_unauthorized_user_can_run_board_mark_as_lost_action_on_own_board(): void
    {
        $signedInUser = $this->asRegularUser()->withPermissionsTo('edit own board')->signIn();

        $boardForSignedIn = $this->factory()->for($signedInUser)->create();
        $otherBoard = $this->factory()->create();

        $this->runAction($this->action, $otherBoard)->assertActionUnauthorized();
        $this->runAction($this->action, $boardForSignedIn)->assertActionOk();
        $this->assertSame(BoardStatus::lost, $boardForSignedIn->fresh()->status);
    }
}
